<?php

namespace App\Controllers;

class GenerateController extends CrudBaseController
{
    public function index()
    {
        return $this->jsonResponse('success', 'GenerateController aktif');
    }

    public function generate($tableName)
    {
        if ($tableName === 'table_settings') {
            return $this->generateTableSettings();
        }
        if ($tableName === 'field_settings') {
            return $this->generateFieldSettings();
        }
        if ($tableName === 'toggle_fields') {
            return $this->generateToggleFields();
        }
        if ($tableName === 'menu') {
            return $this->generateMenu();
        }

        return $this->jsonResponse('error', "Fungsi generate untuk tabel '$tableName' belum tersedia.");
    }


    private function generateTableSettings()
    {
        try {
            $tables = $this->db->listTables();

            if (empty($tables)) {
                return $this->jsonResponse('error', 'Tidak ada tabel ditemukan di database.');
            }

            // 🔹 Daftar ikon spesifik untuk tabel tertentu
            $iconMap = [
                'bulan'             => 'fas fa-calendar-alt',
                'dashboard'         => 'fas fa-tachometer-alt',
                'field_settings'    => 'fas fa-sliders-h',
                'guru'              => 'fas fa-chalkboard-teacher',
                'infotabel'         => 'fas fa-info-circle',
                'kelas'             => 'fas fa-school',
                'kode_surat'        => 'fas fa-envelope-open-text',
                'menu'              => 'fas fa-bars',
                'menu_statis'       => 'fas fa-thumbtack',
                'modal'             => 'fas fa-window-maximize',
                'nama_kode_sekolah' => 'fas fa-university',
                'nomor_surat'       => 'fas fa-hashtag',
                'orangtua'          => 'fas fa-user-friends',
                'set_kolom'         => 'fas fa-columns',
                'siswa'             => 'fas fa-user-graduate',
                'table_settings'    => 'fas fa-cog',
                'tahun'             => 'fas fa-clock',
                'tahun_pelajaran'   => 'fas fa-book',
                'toggle_fields'     => 'fas fa-toggle-on',
                'user'              => 'fas fa-user',
                'versi'             => 'fas fa-code-branch',
            ];

            // 🔹 Daftar ikon random tambahan
            $randomIcons = [
                // 📊 Data / Struktur
                'fas fa-database',
                'fas fa-table',
                'fas fa-cubes',
                'fas fa-project-diagram',
                'fas fa-sitemap',

                // 🗂️ File & Folder
                'fas fa-folder',
                'fas fa-folder-open',
                'fas fa-file-alt',
                'fas fa-file-archive',
                'fas fa-file-excel',
                'fas fa-file-code',

                // ⚙️ Tools & Settings
                'fas fa-cog',
                'fas fa-cogs',
                'fas fa-sliders-h',
                'fas fa-wrench',
                'fas fa-tools',

                // 🧭 Navigasi & List
                'fas fa-list',
                'fas fa-list-ul',
                'fas fa-clipboard-list',
                'fas fa-stream',

                // 📈 Statistik & Grafik
                'fas fa-chart-bar',
                'fas fa-chart-pie',
                'fas fa-chart-line',

                // 👥 User / Access
                'fas fa-users',
                'fas fa-user-cog',
                'fas fa-id-badge',
                'fas fa-address-card',
            ];


            // Ambil data yang sudah ada
            $existing = $this->db->table('table_settings')
                ->select('table_name, role')
                ->get()
                ->getResultArray();

            $existingLookup = [];
            foreach ($existing as $row) {
                $existingLookup[$row['table_name']][$row['role']] = true;
            }

            $dataInsert = [];
            $roles = ['admin', 'user']; // bisa ditambah sesuai kebutuhan

            foreach ($tables as $table) {
                foreach ($roles as $role) {
                    if (!isset($existingLookup[$table][$role])) {

                        // 🔹 Pilih ikon: pakai dari map atau random
                        $key = strtolower($table);
                        if (isset($iconMap[$key])) {
                            $icon = $iconMap[$key];
                        } else {
                            $icon = $randomIcons[array_rand($randomIcons)];
                        }

                        $dataInsert[] = [
                            'table_name' => $table,
                            'role'       => $role,
                            'icon'       => $icon,
                            'menu'       => 'aktif',
                            'tambah'     => 'aktif',
                            'print'      => 'aktif',
                            'export'     => 'aktif',
                            'import'     => 'aktif',
                            'view'       => 'aktif',
                            'edit'       => 'aktif',
                            'delete'     => 'aktif',
                            'generate'   => 'aktif',
                            'delete_all' => 'nonaktif',
                        ];
                    }
                }
            }

            if (!empty($dataInsert)) {
                $this->db->table('table_settings')->insertBatch($dataInsert);
            }

            return $this->jsonResponse('success', 'Generate selesai.', [
                'inserted' => count($dataInsert),
                'existing' => count($existing),
                'tables_total' => count($tables)
            ]);
        } catch (\Throwable $e) {
            return $this->jsonResponse('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }



    // jika nama field mengandung "id", akan di set visible = nonaktif
    public function generateFieldSettings()
    {
        $db = \Config\Database::connect();

        try {
            // 1️⃣ Ambil semua tabel di database
            $tables = $db->listTables();
            if (empty($tables)) {
                throw new \Exception('Tidak ada tabel ditemukan di database.');
            }

            $builder = $db->table('field_settings');
            $inserted = 0;

            foreach ($tables as $table) {
                // 2️⃣ Ambil semua kolom dari masing-masing tabel
                $fields = $db->getFieldNames($table);

                foreach ($fields as $field) {
                    // Dua role: admin & user
                    $roles = ['admin', 'user'];

                    foreach ($roles as $role) {
                        $row = [
                            'table_name' => $table,
                            'field_name' => $field,
                            'visible'    => 'aktif',
                            'role'       => $role
                        ];

                        // 🔹 Tambahan logika: jika nama field mengandung "id", set visible = nonaktif
                        if (stripos($field, 'id') !== false) {
                            $row['visible'] = 'nonaktif';
                        }

                        // 3️⃣ Cek apakah data ini sudah ada
                        $exists = $db->table('field_settings')
                            ->where('table_name', $table)
                            ->where('field_name', $field)
                            ->where('role', $role)
                            ->limit(1)
                            ->get()
                            ->getRow();

                        // 4️⃣ Jika belum ada, insert
                        if (!$exists) {
                            $builder->insert($row);
                            $inserted++;
                        }
                    }
                }
            }

            return $this->response->setJSON([
                'status'   => 'success',
                'message'  => 'Generate field_settings berhasil tanpa duplikasi!',
                'inserted' => $inserted
            ]);
        } catch (\Exception $e) {
            log_message('error', 'Generate field_settings error: ' . $e->getMessage());

            return $this->response->setJSON([
                'status'  => 'error',
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ]);
        }
    }


    public function generateToggleFields()
    {
        $db = \Config\Database::connect();

        try {
            // 1️⃣ Ambil semua tabel di database
            $tables = $db->listTables();
            if (empty($tables)) {
                throw new \Exception('Tidak ada tabel ditemukan di database.');
            }

            $builder = $db->table('toggle_fields');
            $inserted = 0;

            // 2️⃣ Loop setiap tabel dan field-nya
            foreach ($tables as $table) {
                $fields = $db->getFieldNames($table);

                foreach ($fields as $field) {
                    // Data untuk dua role: admin & user
                    $rows = [
                        [
                            'table_name' => $table,
                            'field_name' => $field,
                            'toggle_on'  => 'nonaktif',
                            'role'       => 'admin'
                        ],
                        [
                            'table_name' => $table,
                            'field_name' => $field,
                            'toggle_on'  => 'nonaktif',
                            'role'       => 'user'
                        ]
                    ];

                    foreach ($rows as $row) {

                        // 🔹 Tambahan logika baru (tanpa mengubah yang lama)
                        // Jika nama tabel termasuk salah satu berikut, set toggle_on = 'aktif'
                        $specialTables = ['table_settings', 'field_settings', 'toggle_fields'];
                        if (in_array($row['table_name'], $specialTables)) {
                            $row['toggle_on'] = 'aktif';
                        }

                        // 3️⃣ Cek apakah sudah ada data yang sama
                        $exists = $builder
                            ->where('table_name', $row['table_name'])
                            ->where('field_name', $row['field_name'])
                            ->where('role', $row['role'])
                            ->countAllResults();

                        if ($exists == 0) {
                            $builder->insert($row);
                            $inserted++;
                        }
                    }
                }
            }

            // ✅ Berhasil
            return $this->response->setJSON([
                'status'   => 'success',
                'message'  => 'Generate toggle_fields berhasil!',
                'inserted' => $inserted
            ]);
        } catch (\Exception $e) {
            // ❌ Tangani error
            log_message('error', 'Generate toggle_fields error: ' . $e->getMessage());

            return $this->response->setJSON([
                'status'  => 'error',
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ]);
        }
    }


    public function generateInfotabel()
    {
        try {
            $db = \Config\Database::connect();
            $tables = $db->listTables();

            $builder = $db->table('infotabel');

            $info_tabel = '<p class="mb-2" style="font-size: 0.75rem; line-height: 1;">
    Tabel ini menyimpan data penting yang digunakan dalam <strong>pengelolaan dan operasional sistem</strong>.
  </p>
  <ul class="mb-0 ps-3" style="font-size: 0.75rem; line-height: 1;">
    <li>Berisi informasi utama yang mendukung proses kerja aplikasi.</li>
    <li>Dikelola oleh pengguna sesuai dengan <em>hak akses dan peran</em> masing-masing.</li>
    <li>Pastikan data diperbarui secara berkala agar informasi tetap akurat.</li>
  </ul>';
            $info_user = ' <p class="mb-2" style="font-size: 0.75rem; line-height: 1;">
  Berikut panduan singkat dalam menggunakan fitur pada tabel ini:
</p>
<ul class="pl-3 mb-0" style="font-size: 0.75rem; line-height: 1;">
  <li>
    Gunakan tombol 
    <span class="badge bg-success"><i class="fas fa-plus mr-1"></i>Tambah</span> 
    untuk menambahkan data baru ke dalam tabel.
  </li>
  <li>
    Klik tombol 
    <span class="badge bg-warning text-dark"><i class="fas fa-edit mr-1"></i>Edit</span> 
    untuk memperbarui data yang sudah ada.
  </li>
  <li>
    Pastikan perubahan disimpan dengan benar sebelum menutup halaman.
  </li>
</ul>';

            $inserted = 0;

            foreach ($tables as $table) {
                foreach (['admin', 'user'] as $role) {
                    $exists = $builder->where('nama_tabel', $table)
                        ->where('role', $role)
                        ->get()
                        ->getRow();

                    if (!$exists) {
                        $builder->insert([
                            'nama_tabel' => $table,
                            'info_tabel' => $info_tabel,
                            'info_user'  => $info_user,
                            'role'       => $role,
                        ]);
                        $inserted++;
                    }
                }
            }

            return $this->response->setJSON([
                'status'  => 'success',
                'message' => "Generate selesai! ($inserted data baru ditambahkan)"
            ]);
        } catch (\Throwable $e) {
            return $this->response->setJSON([
                'status'  => 'error',
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ]);
        }
    }

    private function generateMenu()
    {
        $db = \Config\Database::connect();
        $builder = $db->table('menu');

        // Ambil semua nama tabel dari database
        $tables = $db->listTables();
        $inserted = 0;

        foreach ($tables as $tbl) {
            // Cek apakah sudah ada di menu
            $exists = $builder->where('table_name', $tbl)->countAllResults();

            if ($exists == 0) {
                $builder->insert([
                    'kategori'   => null,   // default kategori
                    'nomor'      => null,     // biarkan kosong
                    'table_name' => $tbl,
                    'menu_order' => null      // biarkan kosong
                ]);
                $inserted++;
            }
        }

        if ($inserted > 0) {
            return $this->jsonResponse('success', "$inserted tabel berhasil dimasukkan ke menu.");
        } else {
            return $this->jsonResponse('info', 'Semua tabel sudah ada di menu.');
        }
    }
}
